<?php
/**
 * Klasa WP-Cron dla automatycznego harmonogramu
 *
 * @package BossOfToys_Manager
 */

defined('ABSPATH') || exit;

/**
 * Klasa BOT_Cron
 */
class BOT_Cron {

    /**
     * Dostępne moduły do harmonogramu
     */
    private $modules = array(
        'stock_sync'            => 'stock-sync',
        'price_updater'         => 'price-updater',
        'order_forwarder'       => 'order-forwarder',
        'product_adder'         => 'product-adder',
        'product_deleter'       => 'product-deleter',
        'description_generator' => 'description-generator',
    );

    /**
     * Konstruktor
     */
    public function __construct() {
        // Dodaj własne interwały
        add_filter('cron_schedules', array($this, 'add_cron_intervals'));

        // Zarejestruj hooki dla każdego modułu
        foreach ($this->modules as $key => $api_name) {
            add_action('bot_cron_' . $key, array($this, 'run_' . $key));
        }

        // Zaplanuj zadania przy inicjalizacji
        add_action('init', array($this, 'schedule_events'));
    }

    /**
     * Dodaj własne interwały cron
     */
    public function add_cron_intervals($schedules) {
        $schedules['every_5_minutes'] = array(
            'interval' => 300,
            'display'  => __('Co 5 minut', 'bossoftoys-manager'),
        );

        $schedules['every_15_minutes'] = array(
            'interval' => 900,
            'display'  => __('Co 15 minut', 'bossoftoys-manager'),
        );

        $schedules['every_30_minutes'] = array(
            'interval' => 1800,
            'display'  => __('Co 30 minut', 'bossoftoys-manager'),
        );

        return $schedules;
    }

    /**
     * Zaplanuj wydarzenia na podstawie ustawień
     */
    public function schedule_events() {
        $schedule = get_option('bot_schedule', array());

        foreach ($this->modules as $key => $api_name) {
            $hook = 'bot_cron_' . $key;
            $config = $schedule[$key] ?? array('enabled' => false, 'interval' => 'daily');

            if (!empty($config['enabled'])) {
                // Jeśli nie jest zaplanowane, zaplanuj
                if (!wp_next_scheduled($hook)) {
                    wp_schedule_event(time(), $config['interval'], $hook);
                }
            } else {
                // Jeśli wyłączone, usuń z harmonogramu
                $timestamp = wp_next_scheduled($hook);
                if ($timestamp) {
                    wp_unschedule_event($timestamp, $hook);
                }
            }
        }
    }

    /**
     * Przeharmonogramuj wszystkie zadania (po zmianie ustawień)
     */
    public function reschedule_all() {
        // Usuń wszystkie istniejące
        foreach ($this->modules as $key => $api_name) {
            $hook = 'bot_cron_' . $key;
            $timestamp = wp_next_scheduled($hook);
            if ($timestamp) {
                wp_unschedule_event($timestamp, $hook);
            }
        }

        // Zaplanuj ponownie
        $this->schedule_events();
    }

    /**
     * Uruchom moduł przez API
     *
     * @param string $module_key Klucz modułu
     * @param array  $params     Parametry
     */
    private function run_module($module_key, $params = array()) {
        $api = BOT()->api;

        if (!$api->is_configured()) {
            $this->log('API nie jest skonfigurowane - pomijam ' . $module_key);
            return;
        }

        $api_name = $this->modules[$module_key] ?? null;
        if (!$api_name) {
            $this->log('Nieznany moduł: ' . $module_key);
            return;
        }

        $this->log('Uruchamiam moduł (cron): ' . $api_name);

        $result = $api->run_module($api_name, $params);

        if (is_wp_error($result)) {
            $this->log('Błąd: ' . $result->get_error_message(), 'error');
        } else {
            $this->log('Zadanie dodane do kolejki: ' . ($result['job_id'] ?? 'unknown'));
        }
    }

    /**
     * Logowanie
     *
     * @param string $message Wiadomość
     * @param string $level   Poziom (info, error)
     */
    private function log($message, $level = 'info') {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('[BossOfToys Cron] [' . $level . '] ' . $message);
        }
    }

    // ========== Handlery dla poszczególnych modułów ==========

    /**
     * Stock Sync
     */
    public function run_stock_sync() {
        $this->run_module('stock_sync');
    }

    /**
     * Price Updater
     */
    public function run_price_updater() {
        $this->run_module('price_updater');
    }

    /**
     * Order Forwarder
     */
    public function run_order_forwarder() {
        // Dla order forwarder używamy dry_run=false w cron (automatyczne przekazywanie)
        $this->run_module('order_forwarder', array('dry_run' => false));
    }

    /**
     * Product Adder
     */
    public function run_product_adder() {
        $this->run_module('product_adder');
    }

    /**
     * Product Deleter
     */
    public function run_product_deleter() {
        // Dla product deleter używamy dry_run=true domyślnie (bezpieczeństwo)
        $this->run_module('product_deleter', array('dry_run' => true));
    }

    /**
     * Description Generator
     */
    public function run_description_generator() {
        $this->run_module('description_generator');
    }

    // ========== Helpers ==========

    /**
     * Pobierz następne zaplanowane uruchomienie
     *
     * @param string $module_key Klucz modułu
     * @return int|false Timestamp lub false
     */
    public function get_next_scheduled($module_key) {
        return wp_next_scheduled('bot_cron_' . $module_key);
    }

    /**
     * Sprawdź czy moduł jest zaplanowany
     *
     * @param string $module_key Klucz modułu
     * @return bool
     */
    public function is_scheduled($module_key) {
        return (bool) $this->get_next_scheduled($module_key);
    }

    /**
     * Pobierz status wszystkich zadań cron
     *
     * @return array
     */
    public function get_all_status() {
        $status = array();

        foreach ($this->modules as $key => $api_name) {
            $next = $this->get_next_scheduled($key);
            $status[$key] = array(
                'scheduled'  => (bool) $next,
                'next_run'   => $next ? date('Y-m-d H:i:s', $next) : null,
                'api_module' => $api_name,
            );
        }

        return $status;
    }
}
